//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.3

import wesual.Controls 1.0
import wesual.Ui       1.0
import wesual.data.Core 1.0

ListView {
    id : listView

    property int autoScrollMargin : 50

    readonly property alias containsDrag : dropZone.containsDrag
    readonly property alias dragDocument : p_.dragDocument
    readonly property alias dragActive   : p_.dragActive

    signal documentDropped(Document dropped)

    QtObject {
        id : p_

        readonly property int maxContentX :
            (listView.originX + listView.contentWidth + listView.rightMargin)
            - listView.width
        readonly property int maxContentY :
            listView.contentHeight - (listView.height - listView.originY)
            + bottomMargin

        property Document dragDocument
        property bool     dragActive : false
    }

    Behavior on contentX {
        enabled : !p_.dragDocument && orientation === Qt.Horizontal
        NumberAnimation {
            duration : 300
            easing.type : Easing.OutQuad
        }
    }

    Behavior on contentY {
        enabled : !p_.dragDocument && orientation === Qt.Vertical
        NumberAnimation {
            duration : 300
            easing.type : Easing.OutQuad
        }
    }

    interactive : false
    orientation : Qt.Horizontal
    clip : true
    boundsBehavior : Flickable.StopAtBounds

    move : Transition {
        SmoothedAnimation {
            properties : "x,y"
            duration: 200
        }
    }
    displaced : move
    remove : Transition {
        NumberAnimation {
            duration : 150
            properties : "opacity,scale"
            from : 1
            to : 0.01
            alwaysRunToEnd : true
        }
    }
    add : Transition {
        SequentialAnimation {
            PropertyAction {
                property : "opacity"
                value : 0
            }
            PauseAnimation { duration : 100 }
            NumberAnimation {
                duration : 150
                property : "opacity"
                from : 0
                to : 1
                alwaysRunToEnd : true
            }
        }
    }
    populate : add

    NumberAnimation {
        id : scrollAnimation
        target : listView
        property : listView.orientation === Qt.Horizontal ?
                       "contentX" : "contentY"
        easing.type : Easing.Linear
    }

    UiDropZone {
        id : dropZone

        z : 1
        anchors.fill : parent

        DraggableDropHandler {
            property bool removeOnLeave

            key : "we.captivo.DragDelegate"

            function contentPosition(viewPos) {
                var pos = dropZone.mapToItem(
                            listView.contentItem, viewPos.x, viewPos.y);

                return pos;
            }

            function targetItem(contentPos) {
                var item = listView.itemAt(contentPos.x, contentPos.y);
                if (item !== null) {
                    return [
                        listView.indexAt(contentPos.x, contentPos.y),
                        item
                    ];
                }

                if (listView.orientation === Qt.Horizontal) {
                    contentPos.x -= listView.spacing;
                } else {
                    contentPos.y -= listView.spacing;
                }

                item = listView.itemAt(contentPos.x, contentPos.y);
                if (item !== null) {
                    return [
                        listView.indexAt(contentPos.x, contentPos.y),
                        item
                    ];
                }

                if (listView.orientation === Qt.Horizontal) {
                    contentPos.x += listView.spacing * 2;
                } else {
                    contentPos.y += listView.spacing * 2;
                }

                item = listView.itemAt(contentPos.x, contentPos.y);
                if (item !== null) {
                    return [
                        listView.indexAt(contentPos.x, contentPos.y),
                        item
                    ];
                }

                return [ -1, null ];
            }

            function insertAsFirst(pos) {
                if (listView.orientation === Qt.Horizontal) {
                    if (pos.x < listView.originX)
                        return true;
                } else {
                    if (pos.y < listView.originY)
                        return true;
                }

                return false;
            }

            function insertAsLast(pos) {
                if (listView.orientation === Qt.Horizontal) {
                    if ((pos.x - listView.originX) > listView.contentWidth)
                        return true;
                } else {
                    if ((pos.y - listView.originY) > listView.contentHeight)
                        return true;
                }

                return false;
            }

            function autoScroll(pos) {
                var from, to, scroll = false;

                var scrollHorizontal =
                        listView.orientation === Qt.Horizontal &&
                        listView.visibleArea.widthRatio < 1;
                var scrollVertical =
                        listView.orientation === Qt.Vertical &&
                        listView.visibleArea.heightRatio < 1
                var margin = listView.autoScrollMargin;

                if (scrollHorizontal) {
                    if (pos.x < listView.width && pos.x > listView.width - margin) {
                        from = listView.contentX;
                        to = p_.maxContentX;
                        scroll = true;
                    } else if (pos.x > 0 && pos.x < margin) {
                        from = listView.contentX;
                        to = listView.originX - listView.leftMargin;
                        scroll = true;
                    }
                } else if (scrollVertical) {
                    if (pos.y < listView.height && pos.y > listView.height - margin) {
                        from = listView.contentY;
                        to = p_.maxContentY;
                        scroll = true;
                    } else if (pos.y > 0 && pos.y < margin) {
                        from = listView.contentY;
                        to = listView.originY - listView.topMargin;
                        scroll = true;
                    }
                }

                if (scroll && !scrollAnimation.running) {
                    scrollAnimation.from = from;
                    scrollAnimation.to = to;
                    scrollAnimation.duration = Math.abs(to - from);
                    scrollAnimation.start();
                }

                return scroll;
            }

            onEntered : {
                removeOnLeave = false;
                if (model && model.isModifyable) {
                    var doc = drag.source.document;
                    var currentIndex = model.indexOf(doc);

                    if (currentIndex !== -1) {
                        drag.source.originatingIndex = currentIndex;
                    }
                    p_.dragDocument = doc;
                }
            }

            onPositionChanged : {
                if (!model || !model.isModifyable)
                    return;

                // Autoscroll
                if (autoScroll(drag.position))
                    return;

                scrollAnimation.stop();

                var pos = contentPosition(drag.position);

                var item = targetItem(pos);
            }

            onExited : {
                if (!model.isModifyable)
                    return;

                scrollAnimation.stop();

                if (removeOnLeave) {
                    model.removeElement(drag.source.document);
                } else {
                    model.insertElement(drag.source.document,
                                        drag.source.originatingIndex);
                }

                p_.dragDocument = null;
            }

            onDropped : {
                var insert = true;
                if (autoScroll(drag.position)) {
                    insert = false;
                }

                scrollAnimation.stop();
                var doc = p_.dragDocument;
                p_.dragDocument = null;

                if (!insert)
                    return;

                var pos = contentPosition(drag.position);
                var index = listView.indexAt(pos.x, pos.y);
                insert = false;

                if (index === -1) {
                    if (insertAsFirst(pos)) {
                        index = 0;
                        insert = true;
                    } else if (insertAsLast(pos)) {
                        index = -1;
                        insert = true;
                    }
                } else {
                    insert = true;
                }

                if (insert) {
                    model.insertElement(drag.source.document, index);
                }

                listView.documentDropped(doc);
            }
        }
    }
}
